/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public Liense as published by        *
 * the Free Software Foundation, either version 2 of the License, or (at      * 
 * your option) any later version.                                            *
 *                                                                            *
 * The ITX package is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY *
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
 * for more details.                                                          * 
 *                                                                            *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.                                                   *
 *                                                                            * 
 * Contact information:                                                       *
 * Donna Bergmark                                                             *
 * 484 Rhodes Hall                                                            *
 * Cornell University                                                         *
 * Ithaca, NY 14853-3801                                                      *
 * bergmark@cs.cornell.edu                                                    *
 ******************************************************************************/
package client;

import shared.*;
import cnrg.itx.datax.*;

/**
 * The <code>ClientPresentationControl</code> interface is an interface to 
 * a audio/visual slide presentation controller located on a SPOT client.
 * 
 * @author Jason Howes
 * @see cnrg.apps.spot.client.ClientException
 * @see cnrg.apps.spot.shared.PowerPointControlException
 * @version 1.0, 12/12/1998
 */
public interface ClientPresentationControl
{
	/**
	 * PAM control flag.
	 */
	public static final int PAM_CONTROL = 0;
	
	/**
	 * SPOT control flag.
	 */
	public static final int SPOT_CONTROL = 1;
	
	/**
	 * Various exception messages.
	 */
	public static final String NO_SUCH_TOPIC				= "No such topic found";
	public static final String PRESENTATION_IN_PROGRESS		= "Presentation in progress";
	public static final String PRESENTATION_NOT_IN_PROGRESS	= "Presentation not in progress";

	/**
	 * Starts the presentation.
	 * 
	 * @param PPTfilename name of the PPT presentation file.
	 * @param w the horizontal size of the PPT presentation window (in pixels).
	 * @param h the vertical size of the PPT presentation window (in pixels).
	 * @param x the horizontal screen position of the PPT presentation window.
	 * @param y the vertical screen position of the PPT presentation window.
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code>, 
	 * <code>PowerPointControlException</code>, <code>DataException</code> on error.
	 */
	public void startPresentation(String filename,
		int w, 
		int h, 
		float x, 
		float y) throws ClientException, ClientSessionException, PowerPointControlException, DataException;
	
	/**
	 * Stops the slide presentation.
	 * 
	 * @throws <code>ClientSessionException</code> on error.
	 */
	public void stopPresentation() throws ClientSessionException;
	
	/**
	 * Synchronizes the PPT presentation and audio with the first 
	 * presentation slide.
	 * 
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */
	public void gotoFirstPresentationSlide() throws ClientException, ClientSessionException;
	
	/**
	 * Synchronizes the PPT presentation and audio with the last 
	 * presentation slide.
	 * 
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */
	public void gotoLastPresentationSlide() throws ClientException, ClientSessionException;	
	
	/**
	 * Synchronizes the PPT presentation and audio with the next 
	 * presentation slide.
	 * 
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */
	public void gotoNextPresentationSlide() throws ClientException, ClientSessionException;
	
	/**
	 * Synchronizes the PPT presentation and audio with the previous 
	 * presentation slide.
	 * 
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */
	public void gotoPreviousPresentationSlide() throws ClientException, ClientSessionException;	
	
	/**
	 * Synchronizes the PPT presentation and audio with a specified
	 * presentation topic.
	 * 
	 * @param topic the desired topic with which to synchronize.
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */	
	public void gotoPresentationTopic(String topic) throws ClientException, ClientSessionException;	

	/**
	 * Pauses the slide presentation.
	 * 
	 * @throws <code>ClientSessionException</code> on error.
	 */
	public void pausePresentation() throws ClientSessionException;

	/**
	 * Resumes a paused slide presentation.
	 * 
	 * @throws <code>ClientException</code>, <code>ClientSessionException</code> on error.
	 */
	public void resumePresentation() throws ClientException, ClientSessionException;
}

